const express = require('express');
const cors = require('cors');
const mysql = require('mysql2');
const bcrypt = require('bcryptjs');
const jwt = require('jsonwebtoken');
const bodyParser = require('body-parser');
const { body, validationResult } = require('express-validator');
require('dotenv').config();

const app = express();
app.use(bodyParser.json());
app.use(cors({
    origin: 'http://localhost:3000', // Replace with your frontend URL
    credentials: true,
}));
app.use(express.json());

// JWT Secret Key
const SECRET_KEY = process.env.SECRET_KEY || 'your_jwt_secret_key';

// MySQL Database Connection
const db = mysql.createConnection({
    host: 'localhost',
    user: 'landscaping_user',
    password: 'password123',
    database: 'landscaping',
});

db.connect((err) => {
    if (err) {
        console.error('Database connection failed:', err);
        return;
    }
    console.log('Connected to MySQL database');
});

// JWT Middleware
const authenticateToken = (req, res, next) => {
    const authHeader = req.headers['authorization'];
    const token = authHeader && authHeader.split(' ')[1];

    if (!token) {
        return res.status(401).json({ error: 'Access token required' });
    }

    jwt.verify(token, SECRET_KEY, (err, user) => {
        if (err) {
            if (err.name === 'TokenExpiredError') {
                console.error('Token expired:', err);
                return res.status(401).json({ error: 'Token expired' });
            }
            console.error('Invalid token:', err);
            return res.status(403).json({ error: 'Invalid token' });
        }
        req.user = user; // Attach user info to request object
        next();
    });
};

// User Registration
app.post(
    '/register',
    [
        body('username').notEmpty().withMessage('Username is required'),
        body('email').isEmail().withMessage('Valid email is required'),
        body('password').isLength({ min: 6 }).withMessage('Password must be at least 6 characters'),
    ],
    (req, res) => {
        const errors = validationResult(req);
        if (!errors.isEmpty()) {
            return res.status(400).json({ errors: errors.array() });
        }

        const { username, password, email } = req.body;
        const hashedPassword = bcrypt.hashSync(password, 10);

        const query = 'INSERT INTO users (username, password, email, isAdmin) VALUES (?, ?, ?, ?)';
        db.query(query, [username, hashedPassword, email, false], (err, result) => {
            if (err) {
                if (err.code === 'ER_DUP_ENTRY') {
                    return res.status(400).json({ error: 'Username or email already exists' });
                }
                return res.status(500).json({ error: err.message });
            }
            res.status(201).json({ message: 'User registered successfully' });
        });
    }
);

// User Login
// User Login
app.post('/login', (req, res) => {
    const { username, password } = req.body;

    const query = 'SELECT * FROM users WHERE username = ?';
    db.query(query, [username], (err, results) => {
        if (err) {
            return res.status(500).json({ error: 'Internal server error' });
        }
        if (results.length === 0) {
            return res.status(404).json({ error: 'User not found' });
        }

        const user = results[0];
        const isValid = bcrypt.compareSync(password, user.password);

        if (isValid) {
            const payload = { id: user.id, username: user.username, isAdmin: user.isAdmin };

            // Generate access token and refresh token
            const accessToken = jwt.sign(payload, SECRET_KEY, { expiresIn: '1h' });
            const refreshToken = jwt.sign(payload, SECRET_KEY, { expiresIn: '7d' });

            res.json({ message: 'Login successful', accessToken, refreshToken });
        } else {
            res.status(401).json({ error: 'Invalid password' });
        }
    });
});

// Clients API
app.use('/clients', authenticateToken);

app.get('/clients', (req, res) => {
    const userId = req.user.id;
    const query = 'SELECT * FROM clients WHERE user_id = ?';

    db.query(query, [userId], (err, results) => {
        if (err) {
            return res.status(500).json({ error: err.message });
        }
        res.json(results);
    });
});

app.post(
    '/clients',
    [
        body('name').notEmpty().withMessage('Name is required'),
        body('address').notEmpty().withMessage('Address is required'),
        body('contact').notEmpty().withMessage('Contact is required'),
    ],
    (req, res) => {
        const errors = validationResult(req);
        if (!errors.isEmpty()) {
            return res.status(400).json({ errors: errors.array() });
        }

        const { name, address, contact } = req.body;
        const userId = req.user.id;

        const query = 'INSERT INTO clients (name, address, contact, user_id) VALUES (?, ?, ?, ?)';
        db.query(query, [name, address, contact, userId], (err, result) => {
            if (err) {
                return res.status(500).json({ error: err.message });
            }
            res.status(201).json({ message: 'Client added successfully', clientId: result.insertId });
        });
    }
);

// Refresh token endpoint
app.post('/refresh-token', (req, res) => {
    const { refreshToken } = req.body;

    if (!refreshToken) {
        return res.status(401).json({ error: 'Refresh token required' });
    }

    // Verify the refresh token
    jwt.verify(refreshToken, SECRET_KEY, (err, user) => {
        if (err) {
            return res.status(403).json({ error: 'Invalid refresh token' });
        }

        // Generate a new access token
        const { id, username, isAdmin } = user;
        const newAccessToken = jwt.sign({ id, username, isAdmin }, SECRET_KEY, { expiresIn: '1h' });

        res.json({ accessToken: newAccessToken });
    });
});

// Start Server
const PORT = process.env.PORT || 5001;
app.listen(PORT, () => {
    console.log(`Server running on port ${PORT}`);
});

app.get('/jobs', authenticateToken, (req, res) => {
    const userId = req.user.id;
    const query = `
        SELECT jobs.*, clients.name AS client_name 
        FROM jobs 
        INNER JOIN clients ON jobs.client_id = clients.id 
        WHERE jobs.user_id = ?`;

    db.query(query, [userId], (err, results) => {
        if (err) {
            console.error('Error fetching jobs:', err);
            return res.status(500).json({ error: err.message });
        }
        res.json(results);
    });
});

app.post('/jobs', authenticateToken, (req, res) => {
    const { client_id, name, description, status, start_date, end_date } = req.body;
    const userId = req.user.id;

    const query = `
        INSERT INTO jobs (user_id, client_id, name, description, status, start_date, end_date)
        VALUES (?, ?, ?, ?, ?, ?, ?)`;

    db.query(
        query,
        [userId, client_id, name, description, status, start_date, end_date],
        (err, result) => {
            if (err) {
                console.error('Error creating job:', err);
                return res.status(500).json({ error: err.message });
            }
            res.status(201).json({ message: 'Job created successfully', jobId: result.insertId });
        }
    );
});

app.get('/jobs/:id', authenticateToken, (req, res) => {
    const { id } = req.params;
    const userId = req.user.id;

    const query = `
        SELECT jobs.*, clients.name AS client_name 
        FROM jobs 
        INNER JOIN clients ON jobs.client_id = clients.id 
        WHERE jobs.id = ? AND jobs.user_id = ?`;

    db.query(query, [id, userId], (err, results) => {
        if (err) {
            console.error('Error fetching job details:', err);
            return res.status(500).json({ error: err.message });
        }
        if (results.length === 0) {
            return res.status(404).json({ error: 'Job not found or unauthorized' });
        }
        res.json(results[0]);
    });
});

app.put('/jobs/:id', authenticateToken, (req, res) => {
    const { id } = req.params;
    const { client_id, name, description, status, start_date, end_date } = req.body;
    const userId = req.user.id;

    const query = `
        UPDATE jobs 
        SET client_id = ?, name = ?, description = ?, status = ?, start_date = ?, end_date = ? 
        WHERE id = ? AND user_id = ?`;

    db.query(
        query,
        [client_id, name, description, status, start_date, end_date, id, userId],
        (err, result) => {
            if (err) {
                console.error('Error updating job:', err);
                return res.status(500).json({ error: err.message });
            }
            if (result.affectedRows === 0) {
                return res.status(404).json({ error: 'Job not found or unauthorized' });
            }
            res.json({ message: 'Job updated successfully' });
        }
    );
});

app.delete('/jobs/:id', authenticateToken, (req, res) => {
    const { id } = req.params;
    const userId = req.user.id;

    const query = 'DELETE FROM jobs WHERE id = ? AND user_id = ?';

    db.query(query, [id, userId], (err, result) => {
        if (err) {
            console.error('Error deleting job:', err);
            return res.status(500).json({ error: err.message });
        }
        if (result.affectedRows === 0) {
            return res.status(404).json({ error: 'Job not found or unauthorized' });
        }
        res.status(204).send();
    });
});

// LEADS API

// Get all leads for the authenticated user
app.get('/leads', authenticateToken, (req, res) => {
    const userId = req.user.id;
    const query = 'SELECT * FROM leads WHERE user_id = ?';
    db.query(query, [userId], (err, results) => {
        if (err) {
            console.error('Error fetching leads:', err);
            return res.status(500).json({ error: err.message });
        }
        res.json(results);
    });
});

// Add a new lead
app.post('/leads', authenticateToken, (req, res) => {
    const { name, phone, email, address } = req.body;
    const userId = req.user.id;
    const query = 'INSERT INTO leads (name, phone, email, address, user_id) VALUES (?, ?, ?, ?, ?)';
    db.query(query, [name, phone, email, address, userId], (err, result) => {
        if (err) {
            console.error('Error adding lead:', err);
            return res.status(500).json({ error: err.message });
        }
        res.status(201).json({ message: 'Lead added successfully', leadId: result.insertId });
    });
});

// Update a lead
app.put('/leads/:id', authenticateToken, (req, res) => {
    const { id } = req.params;
    const { name, phone, email, address } = req.body;
    const userId = req.user.id;
    const query = 'UPDATE leads SET name = ?, phone = ?, email = ?, address = ? WHERE id = ? AND user_id = ?';
    db.query(query, [name, phone, email, address, id, userId], (err, result) => {
        if (err) {
            console.error('Error updating lead:', err);
            return res.status(500).json({ error: err.message });
        }
        if (result.affectedRows === 0) {
            return res.status(404).json({ error: 'Lead not found or unauthorized' });
        }
        res.json({ message: 'Lead updated successfully' });
    });
});

// Delete a lead
app.delete('/leads/:id', authenticateToken, (req, res) => {
    const { id } = req.params;
    const userId = req.user.id;
    const query = 'DELETE FROM leads WHERE id = ? AND user_id = ?';
    db.query(query, [id, userId], (err, result) => {
        if (err) {
            console.error('Error deleting lead:', err);
            return res.status(500).json({ error: err.message });
        }
        if (result.affectedRows === 0) {
            return res.status(404).json({ error: 'Lead not found or unauthorized' });
        }
        res.status(204).send();
    });
});

// Services API Endpoints

// Fetch all services for the logged-in user
app.get('/services', authenticateToken, (req, res) => {
    const userId = req.user.id;
    const query = 'SELECT * FROM services WHERE user_id = ?';

    db.query(query, [userId], (err, results) => {
        if (err) {
            console.error('Error fetching services:', err);
            return res.status(500).json({ error: err.message });
        }
        res.json(results);
    });
});

// Add a new service
app.post('/services', authenticateToken, (req, res) => {
    const { name, description, price, duration } = req.body;
    const userId = req.user.id;

    if (!name || !description || !price || !duration) {
        return res.status(400).json({ error: 'All fields are required' });
    }

    const query = 'INSERT INTO services (name, description, price, duration, user_id) VALUES (?, ?, ?, ?, ?)';
    db.query(query, [name, description, price, duration, userId], (err, result) => {
        if (err) {
            console.error('Error adding service:', err);
            return res.status(500).json({ error: err.message });
        }
        res.status(201).json({ message: 'Service added successfully', serviceId: result.insertId });
    });
});

// Update an existing service
app.put('/services/:id', authenticateToken, (req, res) => {
    const { id } = req.params;
    const { name, description, price, duration } = req.body;
    const userId = req.user.id;

    const query = 'UPDATE services SET name = ?, description = ?, price = ?, duration = ? WHERE id = ? AND user_id = ?';
    db.query(query, [name, description, price, duration, id, userId], (err, result) => {
        if (err) {
            console.error('Error updating service:', err);
            return res.status(500).json({ error: err.message });
        }
        if (result.affectedRows === 0) {
            return res.status(404).json({ error: 'Service not found or unauthorized' });
        }
        res.json({ message: 'Service updated successfully' });
    });
});

// Delete a service
app.delete('/services/:id', authenticateToken, (req, res) => {
    const { id } = req.params;
    const userId = req.user.id;

    const query = 'DELETE FROM services WHERE id = ? AND user_id = ?';
    db.query(query, [id, userId], (err, result) => {
        if (err) {
            console.error('Error deleting service:', err);
            return res.status(500).json({ error: err.message });
        }
        if (result.affectedRows === 0) {
            return res.status(404).json({ error: 'Service not found or unauthorized' });
        }
        res.status(204).send();
    });
});

// Invoices API

// Fetch all invoices for the logged-in user
app.get('/invoices', authenticateToken, (req, res) => {
    const userId = req.user.id;
    const query = `
        SELECT invoices.*, clients.name AS client_name
        FROM invoices
        INNER JOIN clients ON invoices.client_id = clients.id
        WHERE invoices.user_id = ?
        ORDER BY invoices.issue_date DESC;
    `;
    db.query(query, [userId], (err, results) => {
        if (err) {
            console.error('Error fetching invoices:', err);
            return res.status(500).json({ error: 'Failed to fetch invoices' });
        }
        res.json(results);
    });
});

// Add a new invoice
app.post('/invoices', authenticateToken, (req, res) => {
    const { client_id, invoice_number, issue_date, due_date, total_amount } = req.body;
    const userId = req.user.id;

    const query = `
        INSERT INTO invoices (user_id, client_id, invoice_number, issue_date, due_date, total_amount)
        VALUES (?, ?, ?, ?, ?, ?);
    `;
    db.query(query, [userId, client_id, invoice_number, issue_date, due_date, total_amount], (err, result) => {
        if (err) {
            console.error('Error adding invoice:', err);
            return res.status(500).json({ error: 'Failed to add invoice' });
        }
        res.status(201).json({ message: 'Invoice added successfully', invoiceId: result.insertId });
    });
});

// Update an existing invoice
app.put('/invoices/:id', authenticateToken, (req, res) => {
    const { id } = req.params;
    const { client_id, invoice_number, issue_date, due_date, total_amount, status } = req.body;

    const query = `
        UPDATE invoices
        SET client_id = ?, invoice_number = ?, issue_date = ?, due_date = ?, total_amount = ?, status = ?
        WHERE id = ? AND user_id = ?;
    `;
    db.query(query, [client_id, invoice_number, issue_date, due_date, total_amount, status, id, req.user.id], (err, result) => {
        if (err) {
            console.error('Error updating invoice:', err);
            return res.status(500).json({ error: 'Failed to update invoice' });
        }
        res.json({ message: 'Invoice updated successfully' });
    });
});

// Delete an invoice
app.delete('/invoices/:id', authenticateToken, (req, res) => {
    const { id } = req.params;

    const query = `
        DELETE FROM invoices
        WHERE id = ? AND user_id = ?;
    `;
    db.query(query, [id, req.user.id], (err, result) => {
        if (err) {
            console.error('Error deleting invoice:', err);
            return res.status(500).json({ error: 'Failed to delete invoice' });
        }
        res.json({ message: 'Invoice deleted successfully' });
    });
});

// Fetch details of a specific invoice
app.get('/invoices/:id', authenticateToken, (req, res) => {
    const { id } = req.params;

    const query = `
        SELECT invoices.*, clients.name AS client_name, clients.contact AS client_contact
        FROM invoices
        INNER JOIN clients ON invoices.client_id = clients.id
        WHERE invoices.id = ? AND invoices.user_id = ?;
    `;
    db.query(query, [id, req.user.id], (err, results) => {
        if (err) {
            console.error('Error fetching invoice details:', err);
            return res.status(500).json({ error: 'Failed to fetch invoice details' });
        }
        if (results.length === 0) {
            return res.status(404).json({ error: 'Invoice not found' });
        }
        res.json(results[0]);
    });
});

// Get all expenses for a user
app.get('/expenses', authenticateToken, (req, res) => {
    const userId = req.user.id;
    const query = 'SELECT * FROM expenses WHERE user_id = ? ORDER BY date DESC';

    db.query(query, [userId], (err, results) => {
        if (err) {
            return res.status(500).json({ error: err.message });
        }
        res.json(results);
    });
});

// Add a new expense
app.post('/expenses', authenticateToken, (req, res) => {
    const { name, category, date, amount, description } = req.body;
    const userId = req.user.id;

    const query = 'INSERT INTO expenses (user_id, name, category, date, amount, description) VALUES (?, ?, ?, ?, ?, ?)';
    db.query(query, [userId, name, category, date, amount, description], (err, result) => {
        if (err) {
            return res.status(500).json({ error: err.message });
        }
        res.status(201).json({ message: 'Expense added successfully', expenseId: result.insertId });
    });
});

// Update an expense
app.put('/expenses/:id', authenticateToken, (req, res) => {
    const { id } = req.params;
    const { name, category, date, amount, description } = req.body;
    const userId = req.user.id;

    const query = 'UPDATE expenses SET name = ?, category = ?, date = ?, amount = ?, description = ? WHERE id = ? AND user_id = ?';
    db.query(query, [name, category, date, amount, description, id, userId], (err, result) => {
        if (err) {
            return res.status(500).json({ error: err.message });
        }
        if (result.affectedRows === 0) {
            return res.status(404).json({ error: 'Expense not found or unauthorized' });
        }
        res.json({ message: 'Expense updated successfully' });
    });
});

// Delete an expense
app.delete('/expenses/:id', authenticateToken, (req, res) => {
    const { id } = req.params;
    const userId = req.user.id;

    const query = 'DELETE FROM expenses WHERE id = ? AND user_id = ?';
    db.query(query, [id, userId], (err, result) => {
        if (err) {
            return res.status(500).json({ error: err.message });
        }
        if (result.affectedRows === 0) {
            return res.status(404).json({ error: 'Expense not found or unauthorized' });
        }
        res.status(204).send();
    });
});

// Get all estimates for the logged-in user
app.get('/estimates', authenticateToken, (req, res) => {
    const userId = req.user.id;

    const query = `
        SELECT 
            estimates.*, 
            clients.name AS client_name, 
            JSON_ARRAYAGG(
                JSON_OBJECT(
                    'id', estimate_items.id, 
                    'description', estimate_items.description, 
                    'quantity', estimate_items.quantity, 
                    'unit_price', estimate_items.unit_price
                )
            ) AS items
        FROM estimates
        INNER JOIN clients ON estimates.client_id = clients.id
        LEFT JOIN estimate_items ON estimates.id = estimate_items.estimate_id
        WHERE estimates.user_id = ?
        GROUP BY estimates.id;
    `;

    db.query(query, [userId], (err, results) => {
        if (err) {
            console.error('Error fetching estimates:', err);
            return res.status(500).json({ error: 'Error fetching estimates' });
        }
        res.json(results);
    });
});

// Add a new estimate
app.post('/estimates', authenticateToken, (req, res) => {
    const userId = req.user.id;
    const { client_id, date, total_amount, items } = req.body;

    if (!client_id || !date || !total_amount) {
        return res.status(400).json({ error: 'All fields are required' });
    }

    if (!items || items.length === 0) {
        return res.status(400).json({ error: 'At least one item is required for an estimate.' });
    }

    const query = `
        INSERT INTO estimates (client_id, user_id, date, total_amount)
        VALUES (?, ?, ?, ?)
    `;

    db.query(query, [client_id, userId, date, total_amount], (err, result) => {
        if (err) {
            console.error('Error adding estimate:', err);
            return res.status(500).json({ error: 'Error adding estimate' });
        }

        const estimateId = result.insertId;

        // Insert items into estimate_items table
        const itemQueries = items.map((item) => {
            return new Promise((resolve, reject) => {
                const itemQuery = `
                    INSERT INTO estimate_items (estimate_id, description, quantity, unit_price)
                    VALUES (?, ?, ?, ?)
                `;
                db.query(
                    itemQuery,
                    [estimateId, item.description, item.quantity, item.unit_price],
                    (err, result) => {
                        if (err) reject(err);
                        else resolve(result);
                    }
                );
            });
        });

        Promise.all(itemQueries)
            .then(() => {
                res.status(201).json({ message: 'Estimate added successfully', estimateId });
            })
            .catch((err) => {
                console.error('Error adding estimate items:', err);
                res.status(500).json({ error: 'Error adding estimate items' });
            });
    });
});

// Update an estimate
app.put('/estimates/:id', authenticateToken, (req, res) => {
    const { id } = req.params;
    const { client_id, date, total_amount, items } = req.body;
    const userId = req.user.id;

    if (!client_id || !date || !total_amount) {
        return res.status(400).json({ error: 'All fields are required' });
    }

    const query = `
        UPDATE estimates
        SET client_id = ?, date = ?, total_amount = ?
        WHERE id = ? AND user_id = ?
    `;

    db.query(query, [client_id, date, total_amount, id, userId], (err, result) => {
        if (err) {
            console.error('Error updating estimate:', err);
            return res.status(500).json({ error: 'Error updating estimate' });
        }
        if (result.affectedRows === 0) {
            return res.status(404).json({ error: 'Estimate not found or unauthorized' });
        }

        // Delete existing items and insert new ones
        const deleteItemsQuery = `DELETE FROM estimate_items WHERE estimate_id = ?`;
        db.query(deleteItemsQuery, [id], (err) => {
            if (err) {
                console.error('Error deleting existing estimate items:', err);
                return res.status(500).json({ error: 'Error updating estimate items' });
            }

            const itemQueries = items.map((item) => {
                return new Promise((resolve, reject) => {
                    const itemQuery = `
                        INSERT INTO estimate_items (estimate_id, description, quantity, unit_price)
                        VALUES (?, ?, ?, ?)
                    `;
                    db.query(
                        itemQuery,
                        [id, item.description, item.quantity, item.unit_price],
                        (err, result) => {
                            if (err) reject(err);
                            else resolve(result);
                        }
                    );
                });
            });

            Promise.all(itemQueries)
                .then(() => {
                    res.json({ message: 'Estimate updated successfully' });
                })
                .catch((err) => {
                    console.error('Error updating estimate items:', err);
                    res.status(500).json({ error: 'Error updating estimate items' });
                });
        });
    });
});

// Delete an estimate
app.delete('/estimates/:id', authenticateToken, (req, res) => {
    const { id } = req.params;
    const userId = req.user.id;

    // First, delete all related estimate items
    const deleteItemsQuery = 'DELETE FROM estimate_items WHERE estimate_id = ?';
    db.query(deleteItemsQuery, [id], (err) => {
        if (err) {
            console.error('Error deleting estimate items:', err);
            return res.status(500).json({ error: 'Error deleting estimate items' });
        }

        // Then, delete the estimate
        const query = 'DELETE FROM estimates WHERE id = ? AND user_id = ?';
        db.query(query, [id, userId], (err, result) => {
            if (err) {
                console.error('Error deleting estimate:', err);
                return res.status(500).json({ error: 'Error deleting estimate' });
            }
            if (result.affectedRows === 0) {
                return res.status(404).json({ error: 'Estimate not found or unauthorized' });
            }
            res.json({ message: 'Estimate deleted successfully' });
        });
    });
});

// Fetch details of a specific estimate
app.get('/estimates/:id', authenticateToken, (req, res) => {
    const { id } = req.params;
    const userId = req.user.id;

    const query = `
        SELECT estimates.*, clients.name AS client_name
        FROM estimates
        INNER JOIN clients ON estimates.client_id = clients.id
        WHERE estimates.id = ? AND estimates.user_id = ?
    `;

    db.query(query, [id, userId], (err, results) => {
        if (err) {
            console.error('Error fetching estimate details:', err);
            return res.status(500).json({ error: 'Error fetching estimate details' });
        }
        if (results.length === 0) {
            return res.status(404).json({ error: 'Estimate not found or unauthorized' });
        }

        const estimate = results[0];

        // Ensure total_amount is a valid number
        estimate.total_amount = Number(estimate.total_amount) || 0;

        const itemsQuery = `SELECT * FROM estimate_items WHERE estimate_id = ?`;
        db.query(itemsQuery, [id], (err, items) => {
            if (err) {
                console.error('Error fetching estimate items:', err);
                return res.status(500).json({ error: 'Error fetching estimate items' });
            }
            res.json({ ...estimate, items });
        });
    });
});